/******************************************************************************\
**
**     COMPANY:	LDW Software
**
**      MODULE:	Orchard
**
** DESCRIPTION:	Planting, growth and harvesting of trees in the orchard
**
**      AUTHOR:	Adrian Francis
**
\******************************************************************************/

#ifndef ORCHARD_H
#define ORCHARD_H

/******************************************************************************\
** LIBRARY INCLUDES
\******************************************************************************/
#include "ldwImage.h"
#include "PuzzleInterface.h"
#include "../Collectables/CollectableInterface.h"
#include "../World/ContentMap.h"

/******************************************************************************\
** CONSTANT DEFINITIONS
\******************************************************************************/
const int Orchard_c_iMaxTrees = 3;
const int Orchard_c_iFruitDropSpeed = 5;
const int Orchard_c_iInitialFruitsOnTree = 1000;
const int Orchard_c_iMaxFruitsOnTree = 1000;
const int Orchard_c_iMinFruitsForHarvesting = 5;
const int Orchard_c_iFruitValue = 4;
const int Orchard_c_iFruitsPerHour = 32; // was 37 from 0.941 to 0.942 //was 25 was 10 was 30
const unsigned int Orchard_c_uiSproutingTime = 5 * 60;
const unsigned int Orchard_c_uiFullGrowthTime = 8 * 60 * 60;

/******************************************************************************\
** CLASS DEFINITIONS
\******************************************************************************/
class COrchard : public IPuzzle, public ICollectable
{
	friend class CPuzzle;

private:

	//class enumerations
	enum EFruit
	{
		eFruit_Banana = 0,
		eFruit_Mango,
		eFruit_Papaya,

		eFruit_NumberOf
	};

	//class structures
	struct STree
	{
		ldwPoint		m_sPos;
		unsigned int	m_uiTimePlanted;
		unsigned int	m_uiLastFruitTime;
		EFruit			m_eType;
		int				m_iNumFruits;
	};

	struct SFruit
	{
		EFruit			m_eType;
		bool			m_bIsDropping;
		bool			m_bIsOnGround;
		ldwPoint		m_sPos;
		int				m_iHeight;
	};

	struct SSaveState
	{
		SFruit			m_sFruit;
		STree			m_sTree[Orchard_c_iMaxTrees];
		int				m_iNumTrees;
	};

	//class data
	SFruit				m_sFruit;
	STree				m_sTree[Orchard_c_iMaxTrees];
	int					m_iNumTrees;

	ldwImageGrid *		m_pCarriedFruits;
	ldwImage *			m_pDirtMound;
	ldwImageGrid *		m_pFruit[eFruit_NumberOf];
	ldwImage *			m_pTree[eFruit_NumberOf];

	//class methods
	//setup
	public:		COrchard(void);
	public:		void Reset();
	public:		void LoadAssets();
	public:		void ReleaseAssets();
	public:		const int LoadState(const unsigned char *pData);
	public:		const int SaveState(const unsigned char *pData, const int iMaxDataSize) const;

	// process
	public:		void Refresh();
	public:		void Draw() const;
	public:		void Draw(const int iRef) const;
	public:		void Update();
	public:		void Advance(CVillager &Villager, const int iRef);
	public:		void Complete();

	public:		void DropFruit(const ldwPoint sPos, const int iHeight);

	public:		const bool Carry(CVillager &Villager, const ECarrying eCarrying);
	public:		void Drop(CVillager &Villager, const ECarrying eCarrying);

	// access
	public:		const ECarrying GetNearbyCollectable(CVillager &Villager, const ldwPoint sPos);
	public:		const CContentMap::EObject TreeObject(const int iTree) const;
	public:		const bool IsTreeMature(const int iTree) const;
	public:		const int SecondsUntilMature(const int iTree) const;
	public:		const bool TreeExists(const int iTree) const;
	public:		const bool EnoughFruitToHarvest(const int iTree);
public:		void DealWithFruitEmergency(CVillager &Villager, const ECarrying eCarrying);

	public:		const int FruitOnTree(const int iTree) const;
	public:		const ldwPoint GetTreePos(const int iTree) const;
	public:		const int NearestTree(const ldwPoint sPos) const;
	public:		void ResetFruitOnTree(const int iTree);
	public:		void AdjustFruitOnTree(const int iTree, const int amount);
	public:		const int GetFruitOnTree(const int iTree);

};

/******************************************************************************\
** GLOBAL DECLARATIONS
\******************************************************************************/
extern COrchard		Orchard;

#endif

/******************************************************************************\
** END OF FILE
\******************************************************************************/